﻿Imports System.Media
Imports System.IO

Public Class Form1
    Private hitRange As Integer = 20
    Private Const treasureRadius As Integer = 10

    Private treasureCount As Integer = 5

    Private rand As New Random()

    Private foundTreasures As New List(Of Point)
    Private treasures As New List(Of Point)

    Private mapWindow As MapForm
    Private digSound As SoundPlayer
    Private foundSound As SoundPlayer

    Private gameTimer As New Timer()
    Private elapsedSeconds As Integer = 0
    Private score As Integer = 0

    Private lblTime As Label
    Private lblScore As Label
    Private lblLeft As Label

    Private highScoreFile As String = "highscores.txt"

    Private lblBonus As Label
    Private bonusFlashTimer As Timer
    Private bonusFlashVisible As Boolean = True
    Private criticalBonusThreshold As Integer = 1000
    Private bonusPulseTimer As Timer
    Private bonusPulseStep As Integer = 0
    Private pulseIncreasing As Boolean = True
    Private currentPulseInterval As Integer = 80 ' default smooth pulse

    ' --- Consistent UI font for Form1 ---
    Private uiFont As New Font("Segoe UI", 12, FontStyle.Regular)
    Private uiLabelFont As New Font("Segoe UI", 13, FontStyle.Bold) ' slightly larger for labels like Bonus

    Private currentDifficulty As String = "Normal"
    Private basePoints As Integer = 1000

    ' --- Multi-level support ---
    Private currentLevel As Integer = 1
    Private maxLevels As Integer = 5
    Private levelInProgress As Boolean = False

    ' --- Limited clicks system ---
    Private maxClicks As Integer = 30   ' default number of digs per round
    Private remainingClicks As Integer = 30
    Private lblClicks As Label

    ' --- Adaptive difficulty tuning ---
    Private levelDifficultyFactor As Double = 1.0
    Private lastLevelPerformance As String = ""

    ' --- Animated Digging ---
    Private digAnimations As New List(Of (Position As Point, Radius As Integer, StepCount As Integer))
    Private digAnimationTimer As Timer
    Private shovelIcon As Image

    ' --- Treasure Graphics ---
    Private treasureImages As New List(Of Image)
    Private treasureImageAssignments As New Dictionary(Of Point, Image)

    ' --- Map Markers (player clicks) ---
    Private playerClickPoints As New List(Of Point)

    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles MyBase.Load
        Me.Text = "Treasure Hunt!"

        Me.StartPosition = FormStartPosition.CenterScreen
        Me.CenterToScreen()
        Me.MaximizeBox = False
        Me.FormBorderStyle = FormBorderStyle.FixedSingle

        'Apply consistent font to the whole form as fallback
        Me.Font = uiFont

        Me.DoubleBuffered = True
        Me.BackColor = Color.SandyBrown
        Me.Size = New Size(840 + 120, 560)

        ' Difficulty selector
        Dim cmbDifficulty As New ComboBox() With {
    .Name = "cmbDifficulty",
    .Location = New Point(492, 10),
    .DropDownStyle = ComboBoxStyle.DropDownList,
    .Width = 120
}
        cmbDifficulty.Items.AddRange(New String() {"Easy", "Normal", "Hard", "Expert"})

        ' --- Restore last difficulty from My.Settings ---
        Dim savedDifficulty As String = My.Settings.DifficultyLevel
        If Not String.IsNullOrEmpty(savedDifficulty) AndAlso cmbDifficulty.Items.Contains(savedDifficulty) Then
            cmbDifficulty.SelectedItem = savedDifficulty
        Else
            cmbDifficulty.SelectedItem = "Normal"
        End If

        AddHandler cmbDifficulty.SelectedIndexChanged, AddressOf DifficultyChanged
        Controls.Add(cmbDifficulty)

        ' Buttons
        Dim btnBury As New Button() With {.Text = "Bury Treasure", .Location = New Point(10, 10), .AutoSize = True}
        AddHandler btnBury.Click, AddressOf BuryTreasure
        Controls.Add(btnBury)

        Dim btnReset As New Button() With {.Text = "Reset", .Location = New Point(140, 10), .AutoSize = True}
        AddHandler btnReset.Click, AddressOf ResetGame
        Controls.Add(btnReset)

        Dim btnMap As New Button() With {.Text = "Show Map", .Location = New Point(230, 10), .AutoSize = True}
        AddHandler btnMap.Click, AddressOf ShowMap
        Controls.Add(btnMap)

        Dim btnScores As New Button() With {.Text = "View High Scores", .Location = New Point(340, 10), .AutoSize = True}
        AddHandler btnScores.Click, AddressOf ShowHighScores
        Controls.Add(btnScores)

        ' Labels
        lblTime = New Label() With {
    .Text = "Time: 0s",
    .Location = New Point(500 + 120, 14),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblTime)

        lblScore = New Label() With {
    .Text = "Score: 0",
    .Location = New Point(600 + 120, 14),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblScore)

        lblLeft = New Label() With {
    .Text = "Treasures: 0",
    .Location = New Point(720 + 120, 14),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblLeft)

        lblClicks = New Label() With {
    .Text = "Clicks Left: 0",
    .Location = New Point(660 + 120, 36),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblClicks)

        ' Bonus label uses the slightly larger bold label font
        lblBonus = New Label() With {
    .Text = "Bonus: 0",
    .Location = New Point(500 + 120, 34),
    .AutoSize = True,
    .Font = uiLabelFont
}
        lblBonus.ForeColor = Color.DarkGreen
        Controls.Add(lblBonus)

        Dim lblLevel As New Label() With {
    .Text = "Level: 1",
    .Location = New Point(10, 40),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblLevel)

        Dim lblDiff As New Label() With {
    .Text = "Difficulty x1.00",
    .Location = New Point(120, 40),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblDiff)

        gameTimer.Interval = 1000
        AddHandler gameTimer.Tick, AddressOf GameTimer_Tick

        bonusFlashTimer = New Timer() With {.Interval = 300} ' 0.3 seconds flash
        AddHandler bonusFlashTimer.Tick, AddressOf BonusFlash_Tick

        bonusPulseTimer = New Timer() With {.Interval = 80} ' update every 80ms for smoothness
        AddHandler bonusPulseTimer.Tick, AddressOf BonusPulse_Tick

        LoadSounds()
        ' --- Apply settings on load ---
        SetDifficulty(cmbDifficulty.SelectedItem.ToString())

        ' --- Animated Digging Setup ---
        digAnimationTimer = New Timer() With {.Interval = 40} ' 25 FPS
        AddHandler digAnimationTimer.Tick, AddressOf DigAnimation_Tick
        digAnimationTimer.Start()

        ' Optional: Load a shovel icon if available
        If IO.File.Exists("shovel.png") Then
            shovelIcon = Image.FromFile("shovel.png")
        End If

        ' --- Load Treasure Graphics ---
        Dim imageFiles As String() = {"coin.png", "gem.png", "chest.png"}
        For Each imgPath In imageFiles
            If IO.File.Exists(imgPath) Then
                treasureImages.Add(Image.FromFile(imgPath))
            End If
        Next

    End Sub

    Private Sub LoadSounds()
        Try
            If File.Exists("dig.wav") Then
                digSound = New SoundPlayer("dig.wav")
                digSound.Load()
            End If
            If File.Exists("found.wav") Then
                foundSound = New SoundPlayer("found.wav")
                foundSound.Load()
            End If
        Catch
            MessageBox.Show("Sound loading failed.")
        End Try
    End Sub

    Private Sub PlayDigSound()
        Try
            If digSound IsNot Nothing Then digSound.Play() Else SystemSounds.Beep.Play()
        Catch
        End Try
    End Sub

    Private Sub PlayFoundSound()
        Try
            If foundSound IsNot Nothing Then foundSound.Play() Else SystemSounds.Hand.Play()
        Catch
        End Try
    End Sub

    Private Sub BuryTreasure(sender As Object, e As EventArgs)
        If levelInProgress Then
            MessageBox.Show("You’re already playing! Find all treasures to advance.")
            Return
        End If

        If currentLevel = 1 Then
            ' Starting a fresh game
            score = 0
            elapsedSeconds = 0
        End If

        treasures.Clear()
        foundTreasures.Clear()
        levelInProgress = True
        bonusFlashTimer.Stop()
        bonusPulseTimer.Stop()
        lblBonus.ForeColor = Color.DarkGreen
        lblBonus.Visible = True

        UpdateScoreDisplay()
        UpdateTimeDisplay()
        UpdateBonusDisplay()

        ' --- Adaptive difficulty scaling ---
        Dim levelFactor As Double = levelDifficultyFactor

        Dim areaWidth As Integer = Me.ClientSize.Width
        Dim areaHeight As Integer = Me.ClientSize.Height - 50

        ' Base values modified by difficulty factor
        Dim adjustedTreasureCount As Integer = Math.Max(3, CInt(treasureCount * levelFactor))
        Dim adjustedHitRange As Integer = Math.Max(8, CInt(hitRange / levelFactor))
        Dim adjustedMaxClicks As Integer = Math.Max(20, CInt(maxClicks / (levelFactor * 0.9)))

        remainingClicks = adjustedMaxClicks
        UpdateClicksDisplay()

        For i As Integer = 1 To adjustedTreasureCount
            treasures.Add(New Point(rand.Next(50, areaWidth - 50), rand.Next(60, areaHeight)))
        Next

        ' Assign random treasure images
        treasureImageAssignments.Clear()
        If treasureImages.Count > 0 Then
            For Each t In treasures
                treasureImageAssignments(t) = treasureImages(rand.Next(treasureImages.Count))
            Next
        End If

        MessageBox.Show($"Level {currentLevel} started!" & vbCrLf &
                    $"Find {adjustedTreasureCount} treasures.", "Next Level")

        ''For i As Integer = 1 To treasureCount
        ''treasures.Add(New Point(rand.Next(50, areaWidth - 50), rand.Next(60, areaHeight)))
        ''Next

        PlayFoundSound()
        ' Reset click count
        remainingClicks = maxClicks
        UpdateClicksDisplay()
        UpdateLevelDisplay()
        UpdateDifficultyDisplay()

        MessageBox.Show("Treasures buried! Timer started. Now dig and find them!", "Ready")

        UpdateMap()
        UpdateLeftDisplay()
        Me.Invalidate()
        gameTimer.Start()
    End Sub

    Private Sub ResetGame(sender As Object, e As EventArgs)
        levelDifficultyFactor = 1.0
        lastLevelPerformance = ""
        currentLevel = 1
        levelInProgress = False
        treasures.Clear()
        foundTreasures.Clear()
        score = 0
        elapsedSeconds = 0
        UpdateBonusDisplay()
        bonusFlashTimer.Stop()
        bonusPulseTimer.Stop()
        lblBonus.ForeColor = Color.DarkGreen
        lblBonus.Visible = True
        gameTimer.Stop()
        UpdateScoreDisplay()
        UpdateTimeDisplay()
        UpdateLeftDisplay()
        Me.Invalidate()
        UpdateMap()
        remainingClicks = maxClicks
        UpdateClicksDisplay()
    End Sub

    Private Sub ShowMap(sender As Object, e As EventArgs)
        If mapWindow Is Nothing OrElse mapWindow.IsDisposed Then
            mapWindow = New MapForm()
        End If
        UpdateMap()
        mapWindow.Show()
        mapWindow.BringToFront()
    End Sub

    Private Sub UpdateMap()
        If mapWindow IsNot Nothing AndAlso Not mapWindow.IsDisposed Then
            ' The map area on Form1 (where game map is drawn) is:
            ' x = 0, y = 50, width = client width, height = client height - 60
            Dim sourceRect As New Rectangle(0, 50, Me.ClientSize.Width, Me.ClientSize.Height - 60)

            mapWindow.UpdateMap(
            treasures.Except(foundTreasures).ToList(),
            foundTreasures.ToList(),
            playerClickPoints.ToList(),
            sourceRect
        )
        End If
    End Sub

    Private Sub GameTimer_Tick(sender As Object, e As EventArgs)
        elapsedSeconds += 1
        UpdateTimeDisplay()
        UpdateBonusDisplay()
    End Sub

    Private Sub UpdateTimeDisplay()
        lblTime.Text = $"Time: {elapsedSeconds}s"
    End Sub

    Private Sub UpdateScoreDisplay()
        lblScore.Text = $"Score: {score}"
    End Sub

    Private Sub UpdateLeftDisplay()
        lblLeft.Text = $"Treasures: {treasures.Count - foundTreasures.Count}"
    End Sub

    Private Sub Form1_MouseClick(sender As Object, e As MouseEventArgs) Handles Me.MouseClick
        ' Record click for map
        playerClickPoints.Add(e.Location)
        UpdateMap()

        ' --- Limited Clicks Feature ---
        If treasures.Count = 0 Then
            PlayDigSound()
            Return
        End If

        ' --- Add Dig Animation ---
        digAnimations.Add((e.Location, 5, 0))

        ' Deduct one click per dig
        remainingClicks -= 1
        UpdateClicksDisplay()

        If remainingClicks <= 0 Then
            MessageBox.Show("No more digs left! Round over.", "Out of Clicks")
            EndGame()
            Return
        End If

        PlayDigSound()

        For Each t In treasures
            Dim dx As Integer = e.X - t.X
            Dim dy As Integer = e.Y - t.Y
            Dim distance As Double = Math.Sqrt(dx * dx + dy * dy)
            If distance < hitRange AndAlso Not foundTreasures.Contains(t) Then
                foundTreasures.Add(t)

                ''Dim basePoints As Integer = 1000
                Dim decay As Integer = 10
                Dim pts As Integer = basePoints - (elapsedSeconds * decay)
                If pts < 50 Then pts = 50
                score += pts

                ' 💎 Reward extra digs
                remainingClicks += 3
                If remainingClicks > maxClicks Then remainingClicks = maxClicks
                UpdateClicksDisplay()

                PlayFoundSound()
                MessageBox.Show($"You found a treasure! +{pts} points", "Success")

                Me.Invalidate()
                UpdateMap()
                UpdateScoreDisplay()
                UpdateLeftDisplay()

                If foundTreasures.Count >= treasures.Count Then
                    EndGame()
                    ''gameTimer.Stop()
                    ''MessageBox.Show($"All treasures found! Time: {elapsedSeconds}s | Score: {score}", "Complete")
                    ''SaveHighScore()
                End If
                Exit For
            End If
        Next
    End Sub

    Private Sub UpdateClicksDisplay()
        lblClicks.Text = $"Clicks Left: {remainingClicks}"
    End Sub

    Private Sub SaveHighScore()
        Dim name As String = InputBox("Enter your name for the leaderboard:", "High Score")
        If String.IsNullOrWhiteSpace(name) Then name = "Anonymous"
        Try
            File.AppendAllLines(highScoreFile, {$"{name},{score},{elapsedSeconds},{DateTime.Now},{currentDifficulty}"})
        Catch
            MessageBox.Show("Error saving score.")
        End Try
    End Sub

    Private Sub EndGame()
        gameTimer.Stop()
        levelInProgress = False

        ' --- Analyze player performance ---
        Dim avgTargetTime As Integer = 30 + (currentLevel * 10) ' expected completion time per level
        Dim speedScore As Double = avgTargetTime / Math.Max(1, elapsedSeconds)
        Dim clickEfficiency As Double = remainingClicks / Math.Max(1, maxClicks)

        Dim performanceScore As Double = (speedScore * 0.6) + (clickEfficiency * 0.4)

        ' Determine performance tier and adjust difficulty
        If performanceScore > 1.2 Then
            levelDifficultyFactor *= 1.25
            lastLevelPerformance = "Excellent (Speed & Efficiency)"
        ElseIf performanceScore > 0.9 Then
            levelDifficultyFactor *= 1.15
            lastLevelPerformance = "Good (Balanced)"
        ElseIf performanceScore > 0.7 Then
            levelDifficultyFactor *= 1.05
            lastLevelPerformance = "Fair"
        Else
            levelDifficultyFactor *= 0.9
            lastLevelPerformance = "Needs Improvement"
        End If

        ' Clamp difficulty to a reasonable range
        If levelDifficultyFactor < 0.8 Then levelDifficultyFactor = 0.8
        If levelDifficultyFactor > 2.5 Then levelDifficultyFactor = 2.5

        UpdateDifficultyDisplay()

        If currentLevel < maxLevels Then
            MessageBox.Show($"You cleared Level {currentLevel}!" & vbCrLf &
                    $"Performance: {lastLevelPerformance}" & vbCrLf &
                    $"Next Level Difficulty: {levelDifficultyFactor:F2}×" & vbCrLf &
                    $"Get ready for Level {currentLevel + 1}!", "Level Complete")
            currentLevel += 1
            BuryTreasure(Nothing, Nothing)
            Return
        End If

        Dim playerName As String = InputBox("You found all the treasures! Enter your name:", "Game Over", "Player")
        If String.IsNullOrWhiteSpace(playerName) Then playerName = "Anonymous"

        ' --- Base score accumulated during play ---
        Dim baseScore As Integer = score

        ' --- Nonlinear (exponential) time bonus ---
        ' Formula: timeBonus = maxBonus * e^(-elapsedSeconds / timeConstant)
        ' Fast finishes get large bonuses; slower times drop off smoothly.
        Dim maxBonus As Integer = 4000
        Dim timeConstant As Double = 20.0 ' Larger = slower decay (adjust to taste)
        Dim timeBonus As Integer = CInt(maxBonus * Math.Exp(-elapsedSeconds / timeConstant))

        ' --- Click Bonus (unused digs reward) ---
        ' Multiplier scales slightly with difficulty
        Dim clickBonusPerDig As Integer

        Select Case currentDifficulty
            Case "Easy" : clickBonusPerDig = 30
            Case "Normal" : clickBonusPerDig = 40
            Case "Hard" : clickBonusPerDig = 50
            Case "Expert" : clickBonusPerDig = 60
            Case Else : clickBonusPerDig = 40
        End Select

        Dim clickBonus As Integer = remainingClicks * clickBonusPerDig

        ' --- Total score ---
        Dim finalScore As Integer = baseScore + timeBonus + clickBonus

        ' --- Save high score ---
        Dim recordLine As String = $"{playerName},{finalScore},{elapsedSeconds},{DateTime.Now},{currentDifficulty}"

        Try
            File.AppendAllText(highScoreFile, recordLine & Environment.NewLine)
        Catch ex As Exception
            MessageBox.Show("Could not save high score: " & ex.Message)
        End Try

        ' --- Compute player's rank for display ---
        Dim rank As Integer = 0
        Dim totalPlayers As Integer = 0

        Try
            If File.Exists(highScoreFile) Then
                Dim scores = File.ReadAllLines(highScoreFile)
                Dim scoreList = New List(Of (Player As String, Score As Integer))

                For Each line In scores
                    Dim parts = line.Split(","c)
                    If parts.Length >= 2 Then
                        Dim s As Integer
                        If Integer.TryParse(parts(1), s) Then
                            scoreList.Add((parts(0), s))
                        End If
                    End If
                Next

                totalPlayers = scoreList.Count
                Dim ordered = scoreList.OrderByDescending(Function(x) x.Score).ToList()
                For i As Integer = 0 To ordered.Count - 1
                    If String.Equals(ordered(i).Player, playerName, StringComparison.OrdinalIgnoreCase) AndAlso ordered(i).Score = finalScore Then
                        rank = i + 1
                        Exit For
                    End If
                Next
            End If
        Catch
            rank = 0
        End Try

        ' ✅ Auto-open leaderboard and highlight this player
        Dim hs As New HighScoresForm(highScoreFile, playerName, finalScore)
        hs.ShowDialog()

        ' --- Show detailed results ---
        Dim msg As String = $"Your Score: {finalScore}" & vbCrLf &
                    $"Base Score: {baseScore}" & vbCrLf &
                    $"Time Bonus: +{timeBonus}" & vbCrLf &
                    $"Click Bonus: +{clickBonus} ({remainingClicks} digs left)" & vbCrLf &
                    $"Time: {elapsedSeconds} seconds"

        If rank > 0 Then
            msg &= vbCrLf & $"🏅 You ranked #{rank} out of {totalPlayers}!"
        End If

        MessageBox.Show(msg, "Congratulations!", MessageBoxButtons.OK, MessageBoxIcon.Information)

        remainingClicks = maxClicks
        UpdateClicksDisplay()

        currentLevel = 1
        ResetGame(Nothing, Nothing)
    End Sub

    Private Sub ShowHighScores(sender As Object, e As EventArgs)
        Dim hs As New HighScoresForm(highScoreFile)
        hs.ShowDialog()
    End Sub

    Private Sub Form1_Paint(sender As Object, e As PaintEventArgs) Handles Me.Paint
        Dim g As Graphics = e.Graphics
        For Each t In foundTreasures
            If treasureImageAssignments.ContainsKey(t) Then
                Dim img As Image = treasureImageAssignments(t)
                Dim imgSize As Integer = 24
                g.DrawImage(img, t.X - imgSize \ 2, t.Y - imgSize \ 2, imgSize, imgSize)
            Else
                ' Fallback to gold circle
                g.FillEllipse(Brushes.Gold, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
                g.DrawEllipse(Pens.Black, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
            End If
        Next
        ''For Each t In foundTreasures
        ''g.FillEllipse(Brushes.Gold, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
        ''g.DrawEllipse(Pens.Black, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
        ''Next
        g.DrawRectangle(Pens.Brown, 0, 50, Me.ClientSize.Width - 1, Me.ClientSize.Height - 60)
        ' --- Draw Dig Animations ---
        For Each anim In digAnimations
            Dim alpha As Integer = 255 - (anim.StepCount * 15)
            If alpha < 0 Then alpha = 0

            Dim digPen As New Pen(Color.FromArgb(alpha, 139, 69, 19), 2)
            e.Graphics.DrawEllipse(digPen, anim.Position.X - anim.Radius, anim.Position.Y - anim.Radius, anim.Radius * 2, anim.Radius * 2)
            digPen.Dispose()

            ' Optional shovel icon overlay
            If shovelIcon IsNot Nothing Then
                Dim iconSize As Integer = 24
                e.Graphics.DrawImage(shovelIcon, anim.Position.X - iconSize \ 2, anim.Position.Y - iconSize \ 2, iconSize, iconSize)
            End If
        Next
    End Sub

    Private Sub UpdateBonusDisplay()
        Dim maxBonus As Integer = 4000
        Dim timeConstant As Double = 20.0
        Dim currentBonus As Integer = CInt(maxBonus * Math.Exp(-elapsedSeconds / timeConstant))

        lblBonus.Text = $"Bonus: {currentBonus}"

        ' --- Color + pulse trigger ---
        If currentBonus < criticalBonusThreshold Then
            If Not bonusPulseTimer.Enabled Then
                pulseIncreasing = True
                bonusPulseStep = 150
                bonusPulseTimer.Start()
            End If
        Else
            lblBonus.ForeColor = Color.DarkGreen
            If bonusPulseTimer.Enabled Then bonusPulseTimer.Stop()
        End If
    End Sub

    Private Sub BonusFlash_Tick(sender As Object, e As EventArgs)
        ' Toggle the label’s visibility for blinking effect
        bonusFlashVisible = Not bonusFlashVisible
        lblBonus.Visible = bonusFlashVisible
    End Sub

    Private Sub BonusPulse_Tick(sender As Object, e As EventArgs)
        ' Pulse intensity oscillates between 100–255
        If pulseIncreasing Then
            bonusPulseStep += 15
            If bonusPulseStep >= 255 Then
                bonusPulseStep = 255
                pulseIncreasing = False
            End If
        Else
            bonusPulseStep -= 15
            If bonusPulseStep <= 100 Then
                bonusPulseStep = 100
                pulseIncreasing = True
            End If
        End If

        ' Apply animated red tone (smooth fade between bright and dim)
        lblBonus.ForeColor = Color.FromArgb(bonusPulseStep, 255, 60, 60)
    End Sub

    Private Sub DifficultyChanged(sender As Object, e As EventArgs)
        Dim cmb As ComboBox = DirectCast(sender, ComboBox)
        currentDifficulty = cmb.SelectedItem.ToString()
        SetDifficulty(currentDifficulty)

        ' --- Save the selected difficulty ---
        My.Settings.DifficultyLevel = currentDifficulty
        My.Settings.Save()
    End Sub

    Private Sub SetDifficulty(level As String)
        Select Case level
            Case "Easy"
                treasureCount = 3
                hitRange = 30
                basePoints = 800
                maxClicks = 60
            Case "Normal"
                treasureCount = 5
                hitRange = 20
                basePoints = 1000
                maxClicks = 50
            Case "Hard"
                treasureCount = 7
                hitRange = 15
                basePoints = 1200
                maxClicks = 40
            Case "Expert"
                treasureCount = 10
                hitRange = 10
                basePoints = 1500
                maxClicks = 30
        End Select
        hitRange = hitRange * 2

        remainingClicks = maxClicks
        UpdateClicksDisplay()

        MessageBox.Show($"Difficulty set to {level}!" & vbCrLf &
                    $"Treasures: {treasureCount}" & vbCrLf &
                    $"Hit range: {hitRange}" & vbCrLf &
                    $"Base points: {basePoints}", "Difficulty")
    End Sub

    Private Sub UpdateDifficultyDisplay()
        Dim lbl = Controls.OfType(Of Label)().FirstOrDefault(Function(x) x.Text.StartsWith("Difficulty"))
        If lbl IsNot Nothing Then lbl.Text = $"Difficulty x{levelDifficultyFactor:F2}"
    End Sub

    Private Sub UpdateLevelDisplay()
        Dim lbl = Controls.OfType(Of Label)().FirstOrDefault(Function(x) x.Text.StartsWith("Level:"))
        If lbl IsNot Nothing Then lbl.Text = $"Level: {currentLevel}/{maxLevels}"
    End Sub

    Private Sub DigAnimation_Tick(sender As Object, e As EventArgs)
        If digAnimations.Count > 0 Then
            ' Progress each animation step
            For i As Integer = 0 To digAnimations.Count - 1
                Dim anim = digAnimations(i)
                anim.Radius += 4
                anim.StepCount += 1
                digAnimations(i) = anim
            Next

            ' Remove finished animations (after about 15 frames)
            digAnimations.RemoveAll(Function(a) a.StepCount > 15)

            ' Trigger repaint
            Me.Invalidate(New Rectangle(0, 50, Me.ClientSize.Width, Me.ClientSize.Height - 60))
        End If
    End Sub

End Class

